import pandas as pd
import pexpect
import os
import re
import glob
import shutil
from pathlib import Path

# Read calculation setting
def input(readfile, meltsfile, index):

    df = pd.read_excel(readfile)

    row = df.iloc[index - 1]

    with open(meltsfile, "w") as f:

        f.write(f"Title: {row['Title']}\n")

        oxides = [
        "SiO2","TiO2","Al2O3","Fe2O3","Cr2O3","FeO","MgO",
        "CaO","Na2O","K2O","P2O5","H2O"
        ]
        for ox in oxides:
            f.write(f"Initial Composition: {ox} {row[ox]:.2f}\n")

        # Temperature
        f.write(f"Initial Temperature: {row['Initial T']:.2f}\n")
        f.write(f"Final Temperature: {row['Final T']:.2f}\n")
        f.write(f"Increment Temperature: {row['Increment T']:.2f}\n")

        # Pressure
        f.write(f"Initial Pressure: {row['Initial P']:.2f}\n")
        f.write(f"Final Pressure: {row['Final P']:.2f}\n")
        f.write(f"Increment Pressure: {row['Increment P']:.2f}\n")

        f.write(f"dp/dt: {row['dp/dt']:.2f}\n")
        value = row['log fO2 path']
        if pd.isna(value):
           value = "None"
        else:
           value = str(value)
        f.write(f"log fo2 Path: {value}\n")
        
        f.write(f"Mode: {row['Mode']}\n")

# Run
def calc(fname, version):

     WORKDIR = Path.home() / "alphamelts"
     os.chdir(WORKDIR)

     child = pexpect.spawn("./alphamelts_macos", encoding="utf-8", timeout=300)
     child.logfile = open("alphamelts_log.txt", "w", encoding="utf-8")
     
     if version == 1 : # rhyolite-MELTS (v. 1.0.2)
        child.expect("Change this\\? \\(y or n\\):")
        child.sendline("n")
        
     elif version == 2 : # rhyolite-MELTS (v. 1.1.0)
        child.expect("Change this\\? \\(y or n\\):")
        child.sendline("y")
        child.expect("public release v 1.1.0\\)\\? \\(y or n\\):")
        child.sendline("y")
        
     elif version == 3 : # rhyolite-MELTS (v. 1.2.0)
        child.expect("Change this\\? \\(y or n\\):")
        child.sendline("y")
        child.expect("public release v 1.1.0\\)\\? \\(y or n\\):")
        child.sendline("n")
        child.expect("public release v 1.2.0\\)\\? \\(y or n\\):")
        child.sendline("y")
        
     elif version == 4 : # pMELTS (v. 5.6.1)
        child.expect("Change this\\? \\(y or n\\):")
        child.sendline("y")
        child.expect("public release v 1.1.0\\)\\? \\(y or n\\):")
        child.sendline("n")
        child.expect("public release v 1.2.0\\)\\? \\(y or n\\):")
        child.sendline("n")
        child.expect("public release v 5.6.1\\)\\? \\(y or n\\):")
        child.sendline("y")
        
        
     child.expect("Your choice:")
     child.sendline("1")     # Read MELTS file

     child.expect("MELTS filename:")
     child.sendline(fname)

     child.expect("Your choice:")
     child.sendline("3")     # Single batch calculation

     child.expect("Choose:")
     child.sendline("2")     # Superliquidus starting guess at current temp

     child.expect(["Your choice:", "Exiting alphaMELTS"], timeout=1200)

     if "Your choice:" in child.before:
         child.sendline("X")

     child.close()

     if os.path.exists(fname):
        os.remove(fname)
     else:
        print(f'no {fname}')
    
     print("Finished. ", fname)


# Summary
def summary(folder):
     # フォルダ内の melts.txt を全探索
     file_list = glob.glob(os.path.join(folder, "**", "*.txt"), recursive=True)

     if not file_list:
         raise ValueError("melts.txt が見つかりませんでした")
         
     # ★ ここで自然順にソートする
     def natural_sort_key(s):
         return [
             int(text) if text.isdigit() else text.lower()
             for text in re.split(r'(\d+)', s)
         ]

     file_list = sorted(file_list, key=natural_sort_key)

     rows = []
     output_file = "all_results.xlsx"     

     def parse_log_file(input_file):

         with open(input_file, "r", encoding="utf-8", errors="ignore") as f:
             text = f.read()

         # ---- Activity & Melt fraction
         m = re.search(r'Activity of H2O\s*=\s*([0-9\.Ee+-]+).*?Melt fraction\s*=\s*([0-9\.Ee+-]+)', text, re.DOTALL)
         if not m:
             raise ValueError("Activity/Melt fraction が見つかりません")
         act_val = m.group(1)
         melt_val = m.group(2)

         # ---- P & T
         p = re.search(r'\bP\s*([0-9\.Ee+-]+)\s*\(([^)]+)\)', text)
         t = re.search(r'\bT\s*([0-9\.Ee+-]+)\s*\(([^)]+)\)', text)
         if not p or not t:
             raise ValueError("P/T が見つかりません")

         p_val = p.group(1)
         p_unit = f"({p.group(2)})"
         t_val = t.group(1)
         t_unit = f"({t.group(2)})"

         # ---- Mineral detection (last one wins)
         pattern = re.compile(r'^([A-Za-z0-9_\-]+:)\s*(.*)$', re.MULTILINE)
         mineral_blocks = []

         for m in pattern.finditer(text):
             name = m.group(1)  # plagioclase:
             rest = m.group(2)  # 0.046651 g, composition ...
             mineral_blocks.append((name, rest))

         if not mineral_blocks:
             raise ValueError("Mineral ブロックが見つかりません")

         # 最後の鉱物を採用
         mineral_name, rest = mineral_blocks[-1]

         # ---- rest の解析
         mass_val = ""
         mass_unit = ""
         comp_word = ""
         formula = ""

         # mass
         m2 = re.search(r'([0-9\.Ee+-]+)\s*(g[, ]?)', rest)
         if m2:
             mass_val = m2.group(1)
             mass_unit = m2.group(2).strip()

         # composition
         m3 = re.search(r'\bcomposition\b\s*([A-Za-z0-9\.\-]+.*)', rest)
         if m3:
             comp_word = "composition"
             formula = m3.group(1).strip()

         # ---- 整形（セル順番）
         row = [
             "Activity of H2O =",
             act_val,
             "Melt fraction =",
             melt_val,
             "P",
             p_val,
             p_unit,
             "T",
             t_val,
             t_unit,
             mineral_name,
             mass_val,
             mass_unit,
             comp_word,
             formula
         ]

         return row


     # ---- 全ファイル解析
     for file in file_list:
         print("Parsing:", file)
         try:
             result_row = parse_log_file(file)
             # どのファイルか識別する列を追加
             result_row = [file] + result_row
             rows.append(result_row)
         except Exception as e:
             print("Error:", file, e)
             # エラーを記録
             rows.append([file, "ERROR", str(e)])


     # ---- DataFrame 化
     df = pd.DataFrame(rows)

     # ---- Excel 保存
     df.to_excel(output_file, index=False, header=False)

     print("Saved:", output_file)


